<?php

namespace App\Http\Controllers\Auth;

use App\Rules\Mobile;
use App\User;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Foundation\Auth\RegistersUsers;
use Modules\Province\Entities\Province;
use Modules\Setting\Entities\Setting;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->redirectTo = '/accounts';
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data,
            [
                'first_name' => 'required|string|max:191',
                'last_name'  => 'required|string|max:191',
                'email'      => 'nullable|string|email|max:191|unique:users',
                'password'   => 'required|string|min:6|confirmed',
                'mobile'     => ['required', 'numeric', 'unique:users', new Mobile, ],
                'phone'      => 'nullable|size:11',
                'city'       => 'nullable|exists:cities,id',
                'post_code'  => 'nullable|size:10',
                'address'    => 'nullable',
            ]
        );
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\User
     */
    protected function create(array $data)
    {
        $user = User::create([
            'first_name' => $data['first_name'],
            'last_name'  => $data['last_name'],
            'email'      => isset($data['email']) ? $data['email'] : NULL,
            'mobile'     => $data['mobile'],
            'password'   => Hash::make($data['password']),
            'phone'      => isset($data['phone']) ?? NULL,
        ]);

        if (isset($data['phone'])) {
            $user->addresses()->create([
                'user_id'   => $user->id,
                'name'      => $user->name,
                'phone'     => $data['phone'],
                'city_id'   => $data['city'],
                'address'   => $data['address'],
                'post_code' => $data['post_code'],
            ]);
        }

        return $user;
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\Http\Response
     */
    public function showRegistrationForm()
    {
        $site_settings = Setting::autoload()->pluck('value', 'key')->toArray();
        $provinces  = Province::with('cities')->get()->keyBy('id');

        return view('auth.register', compact('provinces', 'site_settings'));
    }
    
}
