<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\ResetPasswordRequest;
use App\Models\Forget;
use App\Models\PasswordReset;
use App\Rules\Mobile;
use App\Traits\ResetPassword;
use App\User;
use DB;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use Illuminate\Support\MessageBag;
use Modules\Setting\Entities\Setting;

class ResetPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset requests
    | and uses a simple trait to include this behavior. You're free to
    | explore this trait and override any methods you wish to tweak.
    |
    */

    use ResetsPasswords;
    use ResetPassword;

    /**
     * Where to redirect users after resetting their password.
     *
     * @var string
     */
    protected $redirectTo = '/panel';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Display the password reset view for the given token.
     *
     * @param  int  $token
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showResetForm($mobile = null)
    {
        $site_settings = Setting::autoload()->pluck('value', 'key')->toArray();

        return view('auth.passwords.reset', compact('mobile', 'site_settings'));
    }

    /**
     * Reset the given user's password.
     *
     * @param  ResetPasswordRequest  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function reset(Request $request)
    {
        $request->validate([
            'code'     => 'required|size:6|exists:forgets,code',
            'mobile'   => ['required', new Mobile, 'exists:forgets,mobile', 'exists:users,mobile'],
            'password' => ['required', 'string', 'min:6', 'confirmed'],
        ]);
        $passwordReset = Forget::whereCode($request->code)->whereMobile($request->mobile)->first();
        if (!$passwordReset) {
            $this->doneMessage('کد یا موبایل اشتباه است.', 'error');
            return back();
        }
        $user = User::whereMobile($request->mobile)->firstOrFail();
        $passwordReset->delete();
        $this->resetPassword($user, $request->password);
        $this->doneMessage('رمزعبور شما با موفقیت تغییر یافت.');
        return redirect()->route('home');
    }
    
    public function showResetFormEnglish($mobile = null)
    {
        $site_settings = Setting::autoload()->pluck('value', 'key')->toArray();

        return view('auth.english.passwords.reset', compact('mobile', 'site_settings'));
    }
    
    public function resetEnglish(Request $request)
    {
        $request->validate([
            'code'     => 'required|size:6|exists:forgets,code',
            'mobile'   => ['required', new Mobile, 'exists:forgets,mobile', 'exists:users,mobile'],
            'password' => ['required', 'string', 'min:6', 'confirmed'],
        ]);
        $passwordReset = Forget::whereCode($request->code)->whereMobile($request->mobile)->first();
        if (!$passwordReset) {
            $this->doneMessage('The mobile number or identification code is incorrect', 'error');
            return back();
        }
        $user = User::whereMobile($request->mobile)->firstOrFail();
        $passwordReset->delete();
        $this->resetPassword($user, $request->password);
        $this->doneMessage('Password successfully changed.');
        return redirect()->route('home.english');
    }
    
}
