<?php

namespace App\Traits;

use App\Models\Forget;
use App\Models\PasswordReset;
use Carbon\Carbon;

trait ResetPassword
{
    /**
     * Create a new password reset token for the given user,
     * That token will be send to the user with the link to reset password.
     * A new record will be saved in password_resets table.
     *
     * @param  int  $mobile the user mobile number
     * @return PasswordReset
     */
    protected function createToken($mobile)
    {
        $forget = Forget::whereMobile($mobile)->first();

        if ($forget) {
            $now =  Carbon::now();
            if($now->diffInMinutes($forget->created_at) < 3) {
                return null;
            }
            $forget->delete();
        }

        $passwordReset = new Forget([
            'mobile' => $mobile,
            'code'   => Forget::generateCode(),
        ]);
        $passwordReset->save();
        return $passwordReset;
    }

    /**
     * Delete password reset tokens of the given user,
     * The record which was created in the password_resets table will also be deleted.
     *
     * @param  int  $mobile
     * @param  int  $token
     * @return mixed
     */
    protected function deleteToken($mobile, $token)
    {
        return $this->findToken($mobile, $token)->delete();
    }

    /**
     * @param  int  $mobile
     * @param  int  $token
     * @return mixed
     */
    protected function findToken($mobile, $token)
    {
        return ResetPassword::whereMobile($mobile)->whereToken($token)->first();
    }

    /**
     * Validate the given password reset token,
     * This function will check if the token passed in the argument exists for that given user.
     *
     * @param $mobile
     * @param $token
     * @return bool determine whether a matched record found or not?
     */
    protected function tokenExists($mobile, $token) : bool
    {
        return !!$this->findToken($mobile, $token);
    }

    /**
     * Send a password reset link to a user.
     * This function will create code and send an email with the password reset link to the user.
     *
     * @param  int  $mobile
     * @param  int  $code
     */
    protected function sendResetLink($mobile, $code)
    {
        SendSMS($mobile, $code, '167430');
    }
}